<?php
/* --------------------------------------------------------------
   ParcelServiceRepository.php 2020-04-09
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2019 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService\Repository;

use Doctrine\DBAL\ConnectionException;
use Gambio\Admin\ParcelService\Events\AllParcelServicesRequested;
use Gambio\Admin\ParcelService\Events\DeletedMultipleParcelServices;
use Gambio\Admin\ParcelService\Events\DeletedSingleParcelService;
use Gambio\Admin\ParcelService\Events\DeletionOfMultipleParcelServicesRequested;
use Gambio\Admin\ParcelService\Events\DeletionOfSingleParcelServiceRequested;
use Gambio\Admin\ParcelService\Events\FetchedAllParcelServices;
use Gambio\Admin\ParcelService\Events\FetchedSpecificParcelService;
use Gambio\Admin\ParcelService\Events\SpecificParcelServiceRequested;
use Gambio\Admin\ParcelService\Events\StorageOfMultipleParcelServicesRequested;
use Gambio\Admin\ParcelService\Events\StorageOfSingleParcelServiceRequested;
use Gambio\Admin\ParcelService\Events\StoredMultipleParcelServices;
use Gambio\Admin\ParcelService\Events\StoredSingleParcelService;
use Gambio\Admin\ParcelService\Interfaces\ParcelService;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceId;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceIds;
use Gambio\Admin\ParcelService\Interfaces\ParcelServices;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceSqlCriteria;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceSqlPagination;
use Gambio\Admin\ParcelService\Model\ParcelServiceId as ParcelServiceIdModel;
use Gambio\Admin\ParcelService\Model\ParcelServices as ParcelServicesModel;
use InvalidArgumentException;
use Psr\EventDispatcher\EventDispatcherInterface;

/**
 * Class ParcelServiceRepository
 *
 * @package Gambio\Admin\ParcelService\Repository
 */
class ParcelServiceRepository
{
    /**
     * @var ParcelServiceMapper
     */
    private $mapper;
    
    /**
     * @var ParcelServiceReader
     */
    private $reader;
    
    /**
     * @var ParcelServiceWriter
     */
    private $writer;
    
    /**
     * @var EventDispatcherInterface
     */
    private $eventDispatcher;
    
    
    /**
     * ParcelServiceRepository constructor.
     *
     * @param ParcelServiceMapper      $mapper
     * @param ParcelServiceReader      $reader
     * @param ParcelServiceWriter      $writer
     * @param EventDispatcherInterface $eventDispatcher
     */
    public function __construct(
        ParcelServiceMapper $mapper,
        ParcelServiceReader $reader,
        ParcelServiceWriter $writer,
        EventDispatcherInterface $eventDispatcher
    ) {
        $this->mapper          = $mapper;
        $this->reader          = $reader;
        $this->writer          = $writer;
        $this->eventDispatcher = $eventDispatcher;
    }
    
    
    /**
     * @inheritDoc
     */
    public function getAll(
        ParcelServiceSqlCriteria $criteria,
        ParcelServiceSqlPagination $pagination
    ): ParcelServices {
        $event = AllParcelServicesRequested::create($criteria, $pagination);
        $this->eventDispatcher->dispatch($event);
        
        $parcelServices     = [];
        $parcelServicesData = $this->reader->getAllData($criteria, $pagination);
        foreach ($parcelServicesData as $parcelServiceData) {
            $parcelServices[] = $this->mapper->mapParcelService($parcelServiceData);
        }
        
        $event = FetchedAllParcelServices::create(ParcelServicesModel::create(...$parcelServices));
        $this->eventDispatcher->dispatch($event);
        
        return $event->parcelServices();
    }
    
    
    /**
     * @inheritDoc
     */
    public function getTotalCount(ParcelServiceSqlCriteria $criteria): int
    {
        return $this->reader->getTotalCount($criteria);
    }
    
    
    /**
     * @inheritDoc
     */
    public function getById(ParcelServiceId $id): ParcelService
    {
        $event = SpecificParcelServiceRequested::create($id);
        $this->eventDispatcher->dispatch($event);
        
        $parcelService = $this->reader->getById($id);
        
        $event = FetchedSpecificParcelService::create($this->mapper->mapParcelService($parcelService));
        $this->eventDispatcher->dispatch($event);
        
        return $event->parcelService();
    }
    
    
    /**
     * @inheritDoc
     */
    public function store(ParcelService $parcelService): ParcelServiceId
    {
        $event = StorageOfSingleParcelServiceRequested::create($parcelService);
        $this->eventDispatcher->dispatch($event);
        
        if ($parcelService->id() !== null) {
            $this->writer->update($parcelService);
            
            $event = StoredSingleParcelService::create(ParcelServiceIdModel::create($parcelService->id()));
            $this->eventDispatcher->dispatch($event);
            
            return $event->parcelServiceId();
        }
        
        $id = $this->writer->insert($parcelService);
        
        $event = StoredSingleParcelService::create(ParcelServiceIdModel::create($id));
        $this->eventDispatcher->dispatch($event);
        
        return $event->parcelServiceId();
    }
    
    
    /**
     * @inheritDoc
     *
     * @throws ConnectionException
     */
    public function storeMultiple(ParcelServices $parcelServices): ParcelServiceIds
    {
        $ids       = [];
        $operation = null;
        
        /** @var $parcelService ParcelService $parcelService */
        foreach ($parcelServices as $parcelService) {
            if ($parcelService->id() !== null) {
                $ids[]     = $parcelService->id();
                $operation = 'update';
            } elseif ($operation === 'update') {
                throw new InvalidArgumentException('All provided parcel services need to be new or existing.'
                                                   . 'Can not mix store operation for new and existing parcel services.');
            } else {
                $operation = 'insert';
            }
        }
        
        $event = StorageOfMultipleParcelServicesRequested::create($parcelServices);
        $this->eventDispatcher->dispatch($event);
        
        if ($operation === 'update') {
            $this->writer->updateMultiple($parcelServices);
            
            $event = StoredMultipleParcelServices::create($this->mapper->mapParcelServiceIds($ids));
            $this->eventDispatcher->dispatch($event);
            
            return $event->parcelServiceIds();
        }
        
        $ids = $this->writer->insertMultiple($parcelServices);
        
        $event = StoredMultipleParcelServices::create($this->mapper->mapParcelServiceIds($ids));
        $this->eventDispatcher->dispatch($event);
        
        return $event->parcelServiceIds();
    }
    
    
    /**
     * @inheritDoc
     */
    public function delete(ParcelServiceId $id): void
    {
        $event = DeletionOfSingleParcelServiceRequested::create($id);
        $this->eventDispatcher->dispatch($event);
        
        $this->writer->delete($id);
        
        $event = DeletedSingleParcelService::create($id);
        $this->eventDispatcher->dispatch($event);
    }
    
    
    /**
     * @inheritDoc
     *
     * @throws ConnectionException
     */
    public function deleteMultiple(ParcelServiceIds $ids): void
    {
        $event = DeletionOfMultipleParcelServicesRequested::create($ids);
        $this->eventDispatcher->dispatch($event);
        
        $this->writer->deleteMultiple($ids);
        
        $event = DeletedMultipleParcelServices::create($ids);
        $this->eventDispatcher->dispatch($event);
    }
}